import React, {useRef, useState} from 'react'
import {Device} from 'react-native-ble-plx'
import {Button, StyleSheet, Text, View} from "react-native";
import {BLEService} from "../service/BleService";
import {Lynx, Request} from "./gen/interface";
import {Handshake_Request, HardwareType} from "./gen/handshake";
import {Reader} from 'protobufjs';
import {LynxError, lynxErrorToJSON} from "./gen/error";
import {Copilot_Request} from "./gen/copilot";
import {Measure_Request} from "./gen/measure";
import {LatestResult_Request} from "./gen/latest_result";

export type BleDeviceProps = {
    device: Device
    copilotSN: string
    setCopilotSN: React.Dispatch<React.SetStateAction<string>>
}

export function BleDevice({device, copilotSN, setCopilotSN}: BleDeviceProps) {
    const [connected, setConnected] = useState<boolean>(false)
    const requestIndexRef = useRef(0)
    const hardwareTypeRef = useRef(HardwareType.UNKNOWN)
    const [handshakeResult, setHandshakeResult] = useState<string>("")
    const [copilotResult, setCopilotResult] = useState<string>("")
    const [measureResult, setMeasureResult] = useState<string>("")
    const [latestResultResult, setLatestResultResult] = useState<string>("")

    function extractSN() {
        if (typeof device.manufacturerData === "string") {
            const rawManufactureData = atob(device.manufacturerData)
            return rawManufactureData.substring(2)
        }
        return ""
    }

    function hardwareTypeToString(object: HardwareType): string {
        switch (object) {
            case HardwareType.UNKNOWN:
                return "UNKNOWN";
            case HardwareType.TIMING_CUSHION:
                return "压力垫";
            case HardwareType.STRETCH_DETECTOR:
                return "坐位体前屈";
            case HardwareType.JUMP_DETECTOR:
                return "跳远";
            case HardwareType.UNRECOGNIZED:
            default:
                return "UNRECOGNIZED";
        }
    }

    function resultToString(result: number): string {
        switch (hardwareTypeRef.current) {
            case HardwareType.UNKNOWN:
                return "设备类型未知";
            case HardwareType.TIMING_CUSHION:
                return result.toString() + " ms";
            case HardwareType.STRETCH_DETECTOR:
                return result.toString() + " mm";
            case HardwareType.JUMP_DETECTOR:
                return (result >> 16).toString() + " - " + (result & 0xFFFF).toString() + " mm";
            case HardwareType.UNRECOGNIZED:
            default:
                return "UNRECOGNIZED";
        }
    }

    const _handle_response_indication = (lynx: Lynx) => {
        if (lynx.response != undefined) {
            if (lynx.response.index != requestIndexRef.current) {
                console.log("Response index mismatch: ", +lynx.response.index.toString() + " " + requestIndexRef.current.toString());
                return;
            }
            if (lynx.response.handshake != undefined) {
                const handshake = lynx.response.handshake
                if (handshake.error == LynxError.SUCCESS) {
                    setHandshakeResult((handshake.softVersion >> 24).toString() + "." +
                        (handshake.softVersion >> 16 & 0xFF).toString() + "." +
                        (handshake.softVersion >> 8 & 0xFF).toString() + "+" +
                        (handshake.softVersion & 0xFF).toString() + " " +
                        hardwareTypeToString(handshake.hardwareType)
                    )
                    hardwareTypeRef.current = handshake.hardwareType
                }
            } else if (lynx.response.copilot != undefined) {
                const copilot = lynx.response.copilot
                setCopilotResult(lynxErrorToJSON(copilot.error))
            } else if (lynx.response.measure != undefined) {
                const measure = lynx.response.measure
                setMeasureResult(lynxErrorToJSON(measure.error))
            } else if (lynx.response.latestResult != undefined) {
                const latestResult = lynx.response.latestResult
                if (latestResult.error == LynxError.SUCCESS) {
                    setLatestResultResult(resultToString(latestResult.result))
                } else {
                    setLatestResultResult(lynxErrorToJSON(latestResult.error))
                }
            }
        } else if (lynx.indication != undefined) {
            if (lynx.indication.index != requestIndexRef.current) {
                console.log("Indication index mismatch");
                return;
            }
            if (lynx.indication.copilot != undefined) {
                const copilot = lynx.indication.copilot
                setCopilotResult(copilotResult + ", " + lynxErrorToJSON(copilot.error))
            } else if (lynx.indication.measure != undefined) {
                const measure = lynx.indication.measure
                if (measure.error == LynxError.SUCCESS) {
                    setMeasureResult(resultToString(measure.result))
                } else {
                    setMeasureResult(lynxErrorToJSON(measure.error))
                }
            }
        }
    }

    const _connect = () => {
        BLEService.manager.connectToDevice(device.id)
            .then(device => {
                return device.discoverAllServicesAndCharacteristics()
            })
            .then(device => {
                device.monitorCharacteristicForService("07B0C5FE-5285-45A0-9C77-AC03B69E04CA",
                    "6E287A90-1C27-4057-AF80-8F7600736750", (error, characteristic) => {
                        if (error != null) {
                            console.log(error)
                            return;
                        }
                        const value = characteristic?.value
                        if (typeof value === "string") {
                            const bytes = Uint8Array.from(atob(value), c => c.charCodeAt(0))
                            const lynx = Lynx.decode(Reader.create(bytes));
                            _handle_response_indication(lynx)
                        }
                    })
                device.onDisconnected(() => {
                    setConnected(false)
                })
                setHandshakeResult("")
                setCopilotResult("")
                setMeasureResult("")
                setLatestResultResult("")
                hardwareTypeRef.current = HardwareType.UNKNOWN
                setConnected(true)
            })
    }

    const _sendRequest = (data: Uint8Array) => {
        const base64String = btoa(String.fromCharCode(...data));
        BLEService.manager.writeCharacteristicWithResponseForDevice(device.id,
            "07B0C5FE-5285-45A0-9C77-AC03B69E04CA",
            "0A6AC844-2174-4165-84C8-3E5E632A31E4",
            base64String).then(() => {
            console.log("write success")
        })
    }

    const _sendHandshake = () => {
        requestIndexRef.current += 1
        const handshake = Lynx.encode({
            request: Request.create({
                index: requestIndexRef.current,
                handshake: Handshake_Request.create()
            })
        }).finish()
        _sendRequest(handshake)
    }

    const _sendCopilot = () => {
        requestIndexRef.current += 1
        const copilot = Lynx.encode({
            request: Request.create({
                index: requestIndexRef.current,
                copilot: Copilot_Request.create({
                    sn: copilotSN,
                    timeoutInSecond: 30
                })
            })
        }).finish()
        _sendRequest(copilot)
    }

    const _sendMeasure = () => {
        requestIndexRef.current += 1
        const copilot = Lynx.encode({
            request: Request.create({
                index: requestIndexRef.current,
                measure: Measure_Request.create({
                    start: true,
                    timeoutInSecond: 30
                })
            })
        }).finish()
        _sendRequest(copilot)
    }

    const _sendLatestResult = () => {
        requestIndexRef.current += 1
        const copilot = Lynx.encode({
            request: Request.create({
                index: requestIndexRef.current,
                latestResult: LatestResult_Request.create()
            })
        }).finish()
        _sendRequest(copilot)
    }

    const _copilotSwitch = () => {
        return (
            <View>
                <Button title={copilotSN == extractSN() ? "Copilot" : "Main"} disabled={connected} onPress={() => {
                    if (copilotSN == extractSN()) {
                        setCopilotSN("")
                    } else {
                        setCopilotSN(extractSN())
                    }
                }}/>
                <Button title={connected ? "Disconnect" : "Connect"} disabled={copilotSN == extractSN()}
                        onPress={() => {
                            if (connected) {
                                BLEService.manager.cancelDeviceConnection(device.id).then(() => {
                                    setConnected(false)
                                })
                            } else {
                                _connect()
                            }
                        }}/>
            </View>
        );
    }

    const _deviceControl = () => {
        if (connected) {
            return (
                <View>
                    <View style={{flexDirection: 'row', justifyContent: "space-between", alignItems: 'center'}}>
                        <Button title={"Handshake"} onPress={_sendHandshake}/>
                        <Text> {handshakeResult} </Text>
                    </View>
                    <View style={{flexDirection: 'row', justifyContent: "space-between", alignItems: 'center'}}>
                        <Button title={"Copilot"} onPress={_sendCopilot}/>
                        <Text> {copilotResult} </Text>
                    </View>
                    <View style={{flexDirection: 'row', justifyContent: "space-between", alignItems: 'center'}}>
                        <Button title={"Measure"} onPress={_sendMeasure}/>
                        <Text> {measureResult} </Text>
                    </View>
                    <View style={{flexDirection: 'row', justifyContent: "space-between", alignItems: 'center'}}>
                        <Button title={"Latest Result"} onPress={_sendLatestResult}/>
                        <Text> {latestResultResult} </Text>
                    </View>
                </View>
            );
        }
    }

    return (
        <View style={styles.container}>
            <View style={{flexDirection: "row", justifyContent: "space-between"}}>
                <View style={{justifyContent: "center", alignItems: "flex-start"}}>
                    <Text>SN: {extractSN()}</Text>
                    <Text>MTU: {device.mtu.toString()}</Text>
                    <Text>RSSI: {device.rssi}</Text>
                </View>
                <View style={{justifyContent: "center", alignItems: "center"}}>
                    <_copilotSwitch/>
                </View>
            </View>
            <_deviceControl/>
        </View>
    );
}


const styles = StyleSheet.create({
    container: {
        backgroundColor: '#f9c2ff',
        padding: 20,
        marginVertical: 8,
        marginHorizontal: 16,
    },
    title: {
        fontSize: 32,
    },
});