import FontAwesome from "@expo/vector-icons/FontAwesome";
import {BLEService} from "../service/BleService";
import {Text, View} from "react-native";
import {Device} from "react-native-ble-plx";
import React, {useState} from "react";


export type DeviceExtendedByUpdateTime = Device & { updateTimestamp: number }


interface ScanControlProps {
    copilotSN: string
    setFoundDevices: React.Dispatch<React.SetStateAction<DeviceExtendedByUpdateTime[]>>
}

const MIN_TIME_BEFORE_UPDATE_IN_MILLISECONDS = 5000


export default function ScanControl({copilotSN, setFoundDevices}: ScanControlProps) {
    const [isScanning, setIsScanning] = useState<Boolean>(false)

    const isFoundDeviceUpdateNecessary = (currentDevices: DeviceExtendedByUpdateTime[], updatedDevice: Device) => {
        const currentDevice = currentDevices.find(({id}) => updatedDevice.id === id)
        if (!currentDevice) {
            return true
        }
        return currentDevice.updateTimestamp < Date.now()
    }
    const cloneDeep: <T>(objectToClone: T) => T = objectToClone => JSON.parse(JSON.stringify(objectToClone))

    const addFoundDevice = (device: Device) =>
        setFoundDevices(prevState => {
            if (!isFoundDeviceUpdateNecessary(prevState, device)) {
                return prevState
            }
            // deep clone
            const nextState = cloneDeep(prevState)
            const extendedDevice: DeviceExtendedByUpdateTime = {
                ...device,
                updateTimestamp: Date.now() + MIN_TIME_BEFORE_UPDATE_IN_MILLISECONDS
            } as DeviceExtendedByUpdateTime

            const indexToReplace = nextState.findIndex(currentDevice => currentDevice.id === device.id)
            if (indexToReplace === -1) {
                return nextState.concat(extendedDevice)
            }
            nextState[indexToReplace] = extendedDevice
            return nextState
        })

    const _scanButton = () => {
        if (isScanning) {
            return (
                <FontAwesome name="stop" size={24} color="blue" onPress={() => {
                    BLEService.manager.stopDeviceScan().then(() => {
                        setIsScanning(false);
                    });
                }}/>
            );
        } else {
            return (
                <FontAwesome name="play" size={24} color="blue" onPress={() => {
                    setFoundDevices([])
                    BLEService.initializeBLE().then(
                        () => BLEService.scanDevices(addFoundDevice, ["07B0C5FE-5285-45A0-9C77-AC03B69E04CA"], false).then(
                            () => {
                                setIsScanning(true);
                            }
                        )
                    )
                }}/>
            );
        }
    }

    return (
        <View style={{flexDirection: "row", padding: 15, justifyContent: "space-between"}}>
            <Text>Copilot SN: {copilotSN}</Text>
            <_scanButton/>
        </View>
    );
}